#include <windows.h>
#include <winioctl.h>
#include <math.h>
#include <FCNTL.h>
#include <io.h>
#include <stdio.h>

#include "bulkioct.h"
#include "rtusbapi.h"
#include "usb3000.h"

DWORD GetSystemVersionMy(void);

//------------------------------------------------------------------------
// 
//------------------------------------------------------------------------
CRTUSB3000::CRTUSB3000(HINSTANCE hInst)
{
	//   DLL
	hInstance = hInst;
	//  
	hDevice = INVALID_HANDLE_VALUE;
	//       USB
	UsbSpeed = RTUSB3000::INVALID_USB_SPEED;
	//      
	EnableFlashWrite = FALSE;
	//       
	IsServiceFlashWrite = FALSE;
	//     
	EnableTtlOut = FALSE;
	//   size    ( )
	fi.size = sizeof(RTUSB3000::FLASH);
	ap.size = sizeof(RTUSB3000::INPUT_PARS);
	dp.size = sizeof(RTUSB3000::OUTPUT_PARS);
	//         DSP
	ProgramBaseAddress = 0x1000;
	//         DSP
	VarsBaseAddress = RTUSB3000::VarsBaseAddress;
	//     
	InitializeCriticalSection(&cs);
	//  -     ContolPipe
	MAX_USB_BLOCK = 4096;
	//    
	BASE_ERROR_ID = 200;
	//      IDMA DSP
	DSP_DM = 0x4000;
	//      IDMA DSP
	DSP_PM = 0x0000;
	//   
	LastErrorNumber = 0;
}

//------------------------------------------------------------------------
// 
//------------------------------------------------------------------------
CRTUSB3000::~CRTUSB3000() { }




//========================================================================
//        USB3000
//========================================================================

//------------------------------------------------------------------------
//       USB 
//------------------------------------------------------------------------
BOOL WINAPI CRTUSB3000::OpenDevice(WORD VirtualSlot)
{
	//  
	char ModuleName[10];
	//  
	char szDrvName[18];//, slot[4];

	//   ?
	if(hDevice != INVALID_HANDLE_VALUE) { CloseHandle(hDevice); hDevice = INVALID_HANDLE_VALUE; }

	//   Windows
	DWORD SystemVersion = GetWindowsVersion();
	if((SystemVersion == UNKNOWN_WINDOWS_VERSION) ||
   	(SystemVersion == WINDOWS_95) || (SystemVersion == WINDOWS_98_OR_LATER) ||
		(SystemVersion == WINDOWS_NT)) { LastErrorNumber = 1; return FALSE; }

	//    USB
	sprintf(szDrvName, "\\\\.\\Rtecusb%d", VirtualSlot);

	//    
	hDevice = CreateFile(szDrvName, GENERIC_READ | GENERIC_WRITE, 0, NULL,
						                      OPEN_EXISTING, FILE_FLAG_OVERLAPPED, NULL);
	//   
	if(hDevice == INVALID_HANDLE_VALUE)  { LastErrorNumber = 2; return FALSE; }
	//   
	else if(!GetModuleName(ModuleName)) { LastErrorNumber = 45; CloseDevice(); return FALSE; }
	//    USB3000
	else if(strcmp(ModuleName, "USB3000")) { LastErrorNumber = 46; CloseDevice(); return FALSE; }
	//      
	else if(!GetUsbSpeed(&UsbSpeed)) { LastErrorNumber = 56; CloseDevice(); return FALSE; }
	//        USB
	else if(UsbSpeed >= RTUSB3000::INVALID_USB_SPEED) { LastErrorNumber = 57; CloseDevice(); return FALSE; }
	//        DSP
	if(!GET_VAR_WORD(RTUSB3000::D_ENABLE_TTL_OUT, (SHORT *)&EnableTtlOut)) { LastErrorNumber = 19; CloseDevice(); return FALSE; }
	//        
	ap.InputEnabled = FALSE;

	return TRUE;
}

//------------------------------------------------------------------------------
//    
//------------------------------------------------------------------------------
BOOL WINAPI CRTUSB3000::CloseDevice(void)
{
	BOOL status = TRUE;

	if(hDevice != INVALID_HANDLE_VALUE)
	{
		if(!CloseHandle(hDevice)) { LastErrorNumber = 3; status = FALSE; }
		hDevice = INVALID_HANDLE_VALUE;
	}

	return status;
}

//------------------------------------------------------------------------
//   
//------------------------------------------------------------------------
HANDLE WINAPI CRTUSB3000::GetModuleHandle(void) { return hDevice; }

//------------------------------------------------------------------------
//      USB
//------------------------------------------------------------------------
BOOL WINAPI CRTUSB3000::GetUsbSpeed(BYTE * const UsbSpeed)
{
	BOOL status;
	WORD InBuf[4];
	DWORD cbRet=0;

	//   ?
	if(hDevice == INVALID_HANDLE_VALUE) { LastErrorNumber = 4; return FALSE; }

	//    
	EnterCriticalSection(&cs);

	//     
	InBuf[0] = 1; InBuf[1] = V_GET_USB_SPEED; InBuf[2] = 0x0; InBuf[3] = 0x0;
	//   
	status = DeviceIoControl( hDevice, DIOC_SEND_COMMAND,
				            		&InBuf, sizeof(InBuf),
										UsbSpeed, 1/*bytes*/,
                       			&cbRet, NULL);

	//    
	LeaveCriticalSection(&cs);

	//   :)))))
	if(status) return TRUE;
	//   DeviceIoControl()
	else { LastErrorNumber = 5; return FALSE; }
}

//------------------------------------------------------------------------------
//   
//------------------------------------------------------------------------------
BOOL WINAPI CRTUSB3000::GetModuleName(PCHAR const ModuleName)
{
	BOOL status;
	char String[15];
	WORD InBuf[4];
	DWORD cbRet=0;

	//   ?
	if(hDevice == INVALID_HANDLE_VALUE) { LastErrorNumber = 4; return FALSE; }
	//    
	EnterCriticalSection(&cs);

	//     
	InBuf[0] = 1; InBuf[1] = V_GET_MODULE_NAME; InBuf[2] = 0x0; InBuf[3] = 0x0;
	//   
	status = DeviceIoControl( hDevice, DIOC_SEND_COMMAND,
				            		&InBuf, sizeof(InBuf),
										String, 15/*bytes*/,
                       			&cbRet, NULL);

	//    
	LeaveCriticalSection(&cs);

	//  
	strncpy(ModuleName, String, 10);
	//     0 (  )
	ModuleName[9] = '\0';

	//   :)))))
	if(status) return TRUE;
	//   DeviceIoControl()
	else { LastErrorNumber = 5; return FALSE; }
}

//------------------------------------------------------------------------------
//    
//------------------------------------------------------------------------------
BOOL WINAPI CRTUSB3000::GetModuleSerialNumber(PCHAR const SerialNumber)
{
	BOOL status;
	WORD InBuf[4];
	DWORD cbRet=0;

	//   ?
	if(hDevice == INVALID_HANDLE_VALUE) { LastErrorNumber = 4; return FALSE; }
	//    
	EnterCriticalSection(&cs);

	//      
	InBuf[0] = 1; InBuf[1] = V_GET_MODULE_SERIAL_NUMBER; InBuf[2] = 0x0; InBuf[3] = 0x0;
	//   
	status = DeviceIoControl( hDevice, DIOC_SEND_COMMAND,
	                       		&InBuf, sizeof(InBuf),
					SerialNumber, 9/*bytes*/,
	                       		&cbRet, NULL);

	//    
	LeaveCriticalSection(&cs);

	//     0 (  )
	SerialNumber[8] = '\0';

	//   :)))))
	if(status) return TRUE;
	//   DeviceIoControl()
	else { LastErrorNumber = 5; return FALSE; }
}

//------------------------------------------------------------------------
//      
//  RTUSB3000 ( AVR)
//------------------------------------------------------------------------
BOOL WINAPI CRTUSB3000::GetAvrVersion(PCHAR const AvrVersion)
{
	BOOL status;
	char String[15];
	WORD InBuf[4];
	DWORD cbRet=0;

	//   ?
	if(hDevice == INVALID_HANDLE_VALUE) { LastErrorNumber = 4; return FALSE; }
	//    
	EnterCriticalSection(&cs);

	//     
	InBuf[0] = 1; InBuf[1] = V_GET_MODULE_NAME; InBuf[2] = 0x0; InBuf[3] = 0x0;
	//   
	status = DeviceIoControl( hDevice, DIOC_SEND_COMMAND,
				            		&InBuf, sizeof(InBuf),
										String, 15/*bytes*/,
                       			&cbRet, NULL);

	//    
	LeaveCriticalSection(&cs);

	//  
	strncpy(AvrVersion, String+10, 5);
	//     0 (  )
	AvrVersion[4] = '\0';

	//   :)))))
	if(status) return TRUE;
	//   DeviceIoControl()
	else { LastErrorNumber = 5; return FALSE; }
}

//------------------------------------------------------------------------------
//    
//------------------------------------------------------------------------------
BOOL WINAPI CRTUSB3000::ReleaseInstance(void)
{
	BOOL status = TRUE;

	//    
	DeleteCriticalSection(&cs);
	//    
	if(!CloseDevice()) status = FALSE;
	//    
	delete this;

	return status;
}


//------------------------------------------------------------------------
//    DSP
//------------------------------------------------------------------------
BOOL WINAPI CRTUSB3000::RESET_DSP(void)
{
	BOOL status;
	WORD InBuf[4];
	DWORD cbRet=0;

	//   ?
	if(hDevice == INVALID_HANDLE_VALUE) { LastErrorNumber = 4; return FALSE; }
	//    
	EnterCriticalSection(&cs);

	//    DSP 
	InBuf[0] = 0; InBuf[1] = V_RESET_DSP; InBuf[2] = 0x0; InBuf[3] = 0x0;
	//   
	status = DeviceIoControl( hDevice, DIOC_SEND_COMMAND,
	                       		&InBuf, sizeof(InBuf),
                       			NULL, 0/*bytes*/,
                       			&cbRet, NULL);

	//    
	LeaveCriticalSection(&cs);

	//   :)))))
	if(status) return TRUE;
	//   DeviceIoControl()
	else { LastErrorNumber = 5; return FALSE; }
}


//------------------------------------------------------------------------------
//    DSP   RTUSB3000
//------------------------------------------------------------------------------
BOOL WINAPI CRTUSB3000::LOAD_DSP(PCHAR const FileName)
{
	int	FileHandle;
	DWORD	NBytes;
	char  *CodeByte;
	WORD	*CodeWord;
	WORD 	PmBase, PmWords, DmBase, DmWords;
	char 	PatternHeadStr[]="crrtd";
	char	HeadStr[6];

	//   ?
	if(hDevice == INVALID_HANDLE_VALUE) { LastErrorNumber = 4; return FALSE; }
 	//   DSP 
	else if(!RESET_DSP()) { LastErrorNumber = 6; return FALSE; }

	if(FileName == NULL)
	{
		//      DSP
		HRSRC RsRes = FindResource(hInstance, "USB3000DSP", RT_RCDATA);
		if(RsRes == NULL) { LastErrorNumber = 7; return FALSE; }
		//  
		HGLOBAL RcResHandle = LoadResource(hInstance, RsRes);
		if(RcResHandle == NULL) { LastErrorNumber = 8; return FALSE; }
		// 
		char  *pRcData = (char *)LockResource(RcResHandle);
		if(pRcData == NULL) { LastErrorNumber = 9; return FALSE; }
		//   
		NBytes = SizeofResource(hInstance, RsRes);
		if(NBytes == NULL) { LastErrorNumber = 10; return FALSE; }

		//        DSP
		CodeByte = new char[NBytes+2];
		if(CodeByte == NULL)  { LastErrorNumber = 11; return FALSE; }

		//   
		for(DWORD i=0; i < NBytes; i++) CodeByte[i] = pRcData[i];
	}
	else
	{
		//      DSP
		FileHandle=open(FileName, O_BINARY | O_RDONLY);
		if(FileHandle == -1) { LastErrorNumber = 12; return FALSE; }

		//  -       DSP
		NBytes = filelength(FileHandle);
		if((LONG)NBytes == -1) { LastErrorNumber = 13; return FALSE; }

		//     DSP
		CodeByte = new char[NBytes+2];
		if(CodeByte == NULL)  { LastErrorNumber = 11; return FALSE; }

		//    
		if(read(FileHandle, CodeByte, NBytes) == -1) { LastErrorNumber = 14; delete[] CodeByte; return FALSE; }

		//  
		close(FileHandle);
	}

	memcpy(HeadStr, CodeByte, 5); HeadStr[5] = '\0';
	if(strcmp(PatternHeadStr, HeadStr)) { LastErrorNumber = 15; delete[] CodeByte; return FALSE; }

	//   WORD
	CodeWord = (WORD *)CodeByte;

	//    
	PmBase = CodeWord[6];
	// -    
	PmWords = CodeWord[7];
	//    
	DmBase = CodeWord[8];
	// -    
	DmWords = CodeWord[9];

	//     DSP
	if(!PUT_DM_ARRAY((WORD)DmBase, DmWords, (SHORT *)(CodeWord + 2*PmWords + 10))) { delete[] CodeByte;  LastErrorNumber = 16; return FALSE; }

	//    DSP,    01
	if(!PUT_PM_ARRAY((WORD)(PmBase+1), MAKE_PM_BUFFER_RTUSBAPI(PmWords-1, (CodeWord + 12)))) { delete[] CodeByte;  LastErrorNumber = 17; return FALSE; }

	//      DSP, ..   DSP
	if(!PUT_PM_ARRAY((WORD)PmBase, MAKE_PM_BUFFER_RTUSBAPI(1, CodeWord + 10))) { delete[] CodeByte;  LastErrorNumber = 17; return FALSE; }

	//  
	delete[] CodeByte;

	//          DSP
	LONG PBA; if(!GET_PM_WORD(RTUSB3000::D_PROGRAM_BASE_ADDRESS, &PBA)) { LastErrorNumber = 18; return FALSE; }
	ProgramBaseAddress = (WORD)(PBA >> 8);

	return TRUE;
}

//------------------------------------------------------------------------
//    
//------------------------------------------------------------------------
BOOL WINAPI CRTUSB3000::MODULE_TEST(void)
{
	//   ?
	if(hDevice == INVALID_HANDLE_VALUE) { LastErrorNumber = 4; return FALSE; }

	//        DSP
	WORD TMode1; if(!GET_VAR_WORD(RTUSB3000::D_TEST_VAR1, (SHORT *)&TMode1)) { LastErrorNumber = 19; return FALSE; }
	WORD TMode2; if(!GET_VAR_WORD(RTUSB3000::D_TEST_VAR2, (SHORT *)&TMode2)) { LastErrorNumber = 19; return FALSE; }
	if((TMode1 != 0x5555) || (TMode2 != 0xAAAA)) { LastErrorNumber = 20;  return FALSE; }
	else
	{
		WORD Ready;
		WORD TimeOut = 40;

		//   D_TEST_INTR   DSP
		if(!PUT_VAR_WORD(RTUSB3000::D_TEST_INTR_VAR, 0x77BB)) { LastErrorNumber = 21; return FALSE; }

		//     DSP    D_READY  '1'
		do
		{
			Sleep(25);
			if(!GET_VAR_WORD(RTUSB3000::D_MODULE_READY, (SHORT *)&Ready)) { LastErrorNumber = 19; return FALSE; }
		} while((!Ready) && (TimeOut--));
		//   
		if(TimeOut == 0xFFFF)  { LastErrorNumber = 22; return FALSE; }

		//   DSP   
		if(!SEND_COMMAND(C_TEST)) { LastErrorNumber = 23; return FALSE; }

		Sleep(100);

		//   ,    D_TEST_INTR   0xAA55
		WORD TestLoadVar; if(!GET_VAR_WORD(RTUSB3000::D_TEST_INTR_VAR, (SHORT *)&TestLoadVar)) { LastErrorNumber = 19; return FALSE; }
		if(TestLoadVar != 0xAA55) { LastErrorNumber = 24; return FALSE; }
	}

	return TRUE;
}

//------------------------------------------------------------------------
//     DSP
//------------------------------------------------------------------------
BOOL WINAPI CRTUSB3000::GET_DSP_INFO(RTUSB3000::DSP_INFO * const DspInfo)
{
	WORD i;
	WORD dv;

	//   ,    
	//  DSP (9  + '\0')
	//   USB3000    "USB3000"
	for(i = 0x0; i < sizeof(DspInfo->Target)/2; i++)
		if(!GET_VAR_WORD((WORD)(RTUSB3000::D_TARGET + i), (SHORT *)&DspInfo->Target[2*i])) { LastErrorNumber = 19; return FALSE; }

	//     DSP (5  + '\0')
	//   DSP   "rtech"
	for(i = 0x0; i < sizeof(DspInfo->Label)/2; i++)
		if(!GET_VAR_WORD((WORD)(RTUSB3000::D_LABEL + i), (SHORT *)&DspInfo->Label[2*i])) { LastErrorNumber = 19; return FALSE; }

	//    DSP
	if(!GET_VAR_WORD(RTUSB3000::D_VERSION, (SHORT *)&dv)) { LastErrorNumber = 19; return FALSE; }
	DspInfo->DspMajor = (BYTE)(dv >> 8);
	DspInfo->DspMinor = (BYTE)(dv & 0xFF);

	return TRUE;
}

//------------------------------------------------------------------------
//    DSP   
//------------------------------------------------------------------------
BOOL WINAPI CRTUSB3000::SEND_COMMAND(WORD Command)
{
	WORD InBuf[4];
	DWORD cbRet=0;
	WORD TimeOut = 5000;

	//   ?
	if(hDevice == INVALID_HANDLE_VALUE) { LastErrorNumber = 4; return FALSE; }
	//   
	else if(Command >= C_LAST_COMMAND) { LastErrorNumber = 25; return FALSE; }

	//      DSP
	InBuf[0] = 0; InBuf[1] = V_COMMAND_IRQ; InBuf[2] = 0x0; InBuf[3] = 0x0;
	//    DSP   
	if(!PUT_VAR_WORD(RTUSB3000::D_COMMAND, Command)) { LastErrorNumber = 21; return FALSE; }

	//    
	EnterCriticalSection(&cs);

	//      DSP 
	if(!DeviceIoControl(hDevice, DIOC_SEND_COMMAND, &InBuf, sizeof(InBuf), NULL, 0, &cbRet, NULL)) { LastErrorNumber = 5; LeaveCriticalSection(&cs); return FALSE; }

	//    
	LeaveCriticalSection(&cs);

	// ,      DSP,
	// .   D_COMMAND   
	do
	{
	   if(!GET_VAR_WORD(RTUSB3000::D_COMMAND, (SHORT *)&Command)) { LastErrorNumber = 19; return FALSE; }
	} while(Command && (TimeOut--));

	//      
	if(TimeOut == 0xFFFF) { LastErrorNumber = 26; return FALSE; }
	else return TRUE;
}

//------------------------------------------------------------------------------
//         
//------------------------------------------------------------------------------
int WINAPI CRTUSB3000::GetLastErrorString(LPTSTR const lpBuffer, DWORD nSize)
{
	return LoadString(hInstance, BASE_ERROR_ID + LastErrorNumber, lpBuffer, nSize);
}




//========================================================================
//           
//========================================================================

//------------------------------------------------------------------------
//         
//------------------------------------------------------------------------
BOOL WINAPI WINAPI CRTUSB3000::GET_INPUT_PARS(RTUSB3000::INPUT_PARS * const ap)
{
	WORD i, SclockDiv, InterKadrDelay;
	SHORT AdcOffsetCoef;
   WORD  AdcScaleCoef;
	double DspClockout;

	//   ?
	if(hDevice == INVALID_HANDLE_VALUE) { LastErrorNumber = 4; return FALSE; }
	//        RTUSB3000::INPUT_PARS
	else if(this->ap.size != ap->size) { LastErrorNumber = 27; return FALSE; }
	//       RTUSB3000::FLASH (  )
	else if(strcmp((char *)fi.Name, "USB3000")) { LastErrorNumber = 28; return FALSE; }
	//    
	else if(!GET_VAR_WORD(RTUSB3000::D_INPUT_ENABLED, (SHORT *)&ap->InputEnabled)) { LastErrorNumber = 19; return FALSE; }
	//      
	else if(!GET_VAR_WORD(RTUSB3000::D_CORRECTION_ENABLED, (SHORT *)&ap->CorrectionEnabled)) { LastErrorNumber = 19; return FALSE; }
	//       :   
	else if(!GET_VAR_WORD(RTUSB3000::D_INPUT_CLOCK_SOURCE, (SHORT *)&ap->InputClockSource)) { LastErrorNumber = 58; return FALSE; }

	ap->InputEnabled &= 0xFFFFL;
	ap->CorrectionEnabled &= 0xFFFFL;

	//      (  )
	if(!GET_VAR_WORD(RTUSB3000::D_INPUT_TYPE, (SHORT *)&ap->InputType)) { LastErrorNumber = 19; return FALSE; }

	//      
	if(!GET_VAR_WORD(RTUSB3000::D_SYNCHRO_TYPE, (SHORT *)&ap->SynchroType)) { LastErrorNumber = 19; return FALSE; }
	else if(!GET_VAR_WORD(RTUSB3000::D_SYNCHRO_AD_TYPE, (SHORT *)&ap->SynchroAdType)) { LastErrorNumber = 19; return FALSE; }
	else if(!GET_VAR_WORD(RTUSB3000::D_SYNCHRO_AD_MODE, (SHORT *)&ap->SynchroAdMode)) { LastErrorNumber = 19; return FALSE; }
	else if(!GET_VAR_WORD(RTUSB3000::D_SYNCHRO_AD_CHANNEL, (SHORT *)&ap->SynchroAdChannel)) { LastErrorNumber = 19; return FALSE; }
	else if(!GET_VAR_WORD(RTUSB3000::D_SYNCHRO_AD_POROG, (SHORT *)&ap->SynchroAdPorog)) { LastErrorNumber = 19; return FALSE; }

	//  
	if(!GET_VAR_WORD(RTUSB3000::D_CONTROL_TABLE_LENGHT, (SHORT *)&ap->ChannelsQuantity)) { LastErrorNumber = 19; return FALSE; }
	else if(ap->ChannelsQuantity > 128) { LastErrorNumber = 29; return FALSE; }
	for(i=0; i < ap->ChannelsQuantity; i++) if(!GET_VAR_WORD((WORD)(RTUSB3000::D_CONTROL_TABLE + i), (SHORT *)&ap->ControlTable[i])) { LastErrorNumber = 19; return FALSE; }

	//    
	DspClockout = RTUSB3000::DSP_CLOCK_OUT; //   DSP  
	if(fabs(DspClockout - 72000.0) > 0.00001) { LastErrorNumber = 30; return FALSE; }
	if(!GET_VAR_WORD(RTUSB3000::D_INPUT_RATE, (SHORT *)&SclockDiv)) { LastErrorNumber = 19; return FALSE; }
	ap->InputRate = DspClockout/(2.*(WORD)(SclockDiv + 1.0));
	if(!GET_VAR_WORD(RTUSB3000::D_INTER_KADR_DELAY, (SHORT *)&InterKadrDelay)) { LastErrorNumber = 19; return FALSE; }
	ap->InterKadrDelay = (WORD)(InterKadrDelay + 1.)/ap->InputRate; //    
	ap->ChannelRate = 1./((ap->ChannelsQuantity - 1.)/ap->InputRate + ap->InterKadrDelay);

	// FIFO  
	if(!GET_VAR_WORD(RTUSB3000::D_INPUT_FIFO_BASE_ADDRESS, (SHORT *)&ap->InputFifoBaseAddress)) { LastErrorNumber = 19; return FALSE; }
	if(!GET_VAR_WORD(RTUSB3000::D_CUR_INPUT_FIFO_LENGTH, (SHORT *)&ap->InputFifoLength)) { LastErrorNumber = 19; return FALSE; }

	//     
	for(i = 0x0; i < 0x8; i++)
	{
		if(!GET_VAR_WORD((WORD)(RTUSB3000::D_ADC_ZERO + i), (SHORT *)&AdcOffsetCoef)) { LastErrorNumber = 19; return FALSE; }
		ap->AdcOffsetCoef[i] = AdcOffsetCoef;
		if(!GET_VAR_WORD((WORD)(RTUSB3000::D_ADC_SCALE + i), (SHORT *)&AdcScaleCoef)) { LastErrorNumber = 19; return FALSE; }
		ap->AdcScaleCoef[i] = (double)AdcScaleCoef/(double)0x8000;
	}

	//         RTUSB3000::INPUT_PARS
	this->ap = *ap;
	// !!!
	return TRUE;
}

//------------------------------------------------------------------------
//         
//------------------------------------------------------------------------
BOOL WINAPI CRTUSB3000::SET_INPUT_PARS(RTUSB3000::INPUT_PARS * const ap)
{
	WORD i;
	SHORT AdcOffsetCoef;
   WORD  AdcScaleCoef;
	double DspClockout;
	double InputRateLimit;

	//        
	//     RTUSB3000::INPUT_PARS
	this->ap.InputEnabled = FALSE;

	//   ?
	if(hDevice == INVALID_HANDLE_VALUE) { LastErrorNumber = 4; return FALSE; }
	//        RTUSB3000::INPUT_PARS
	else if(this->ap.size != ap->size) { LastErrorNumber = 27; return FALSE; }
	//       RTUSB3000::FLASH (  )
	else if(strcmp((char *)fi.Name, "USB3000")) { LastErrorNumber = 28; return FALSE; }

	//     
	if(!PUT_VAR_WORD(RTUSB3000::D_CORRECTION_ENABLED, (SHORT)(ap->CorrectionEnabled))) { LastErrorNumber = 21; return FALSE; }
	//        :   
	else if(!PUT_VAR_WORD(RTUSB3000::D_INPUT_CLOCK_SOURCE, (SHORT)(ap->InputClockSource))) { LastErrorNumber = 58; return FALSE; }

	//   -     
	if(ap->ChannelsQuantity == 0x0) { LastErrorNumber = 48; return FALSE; }
	else if(ap->ChannelsQuantity > 128) { LastErrorNumber = 49; return FALSE; }
	//    DSP -     
	if(!PUT_VAR_WORD(RTUSB3000::D_CONTROL_TABLE_LENGHT, ap->ChannelsQuantity)) { LastErrorNumber = 21; return FALSE; }

	//     :  ,    
	if(!FindoutInputType(ap)) { LastErrorNumber = 50; return FALSE; }
	//   MIXED_DATA    :(((
	else if((ap->InputType == RTUSB3000::EMPTY_DATA) ||
				(ap->InputType == RTUSB3000::MIXED_DATA)) { LastErrorNumber = 51; return FALSE; }
	else if(!PUT_VAR_WORD(RTUSB3000::D_INPUT_TYPE, ap->InputType)) { LastErrorNumber = 21; return FALSE; }

	//    
	if(!ControlTableCreation(ap)) { LastErrorNumber = 52; return FALSE; }
	//   
	for(i = 0x0; i < ap->ChannelsQuantity; i++)
	   	if(!PUT_VAR_WORD((WORD)(RTUSB3000::D_CONTROL_TABLE + i), (SHORT)ap->ControlTable[i])) { LastErrorNumber = 21; return FALSE; }

	//      
	if(ap->SynchroType >= RTUSB3000::INVALID_INPUT_SYNCHRO) { LastErrorNumber = 31; return FALSE; }
	else if(!PUT_VAR_WORD(RTUSB3000::D_SYNCHRO_TYPE, ap->SynchroType)) { LastErrorNumber = 21; return FALSE; }
	else if(ap->SynchroType == RTUSB3000::ANALOG_SYNCHRO)
	{
	   if(!PUT_VAR_WORD(RTUSB3000::D_SYNCHRO_AD_TYPE, ap->SynchroAdType)) { LastErrorNumber = 21; return FALSE; }
	   if(!PUT_VAR_WORD(RTUSB3000::D_SYNCHRO_AD_MODE, ap->SynchroAdMode)) { LastErrorNumber = 21; return FALSE; }
	   if(!PUT_VAR_WORD(RTUSB3000::D_SYNCHRO_AD_CHANNEL, ap->SynchroAdChannel)) { LastErrorNumber = 21; return FALSE; }
	   if(!PUT_VAR_WORD(RTUSB3000::D_SYNCHRO_AD_POROG, ap->SynchroAdPorog)) { LastErrorNumber = 21; return FALSE; }
	}

	//     ( )      USB
	if(UsbSpeed == RTUSB3000::USB11) InputRateLimit = 500.0;
	else if(UsbSpeed == RTUSB3000::USB20)
	{
		//         ( )
		if(ap->InputType == RTUSB3000::MIXED_DATA) InputRateLimit = 1000.;
		else if(ap->InputType == RTUSB3000::ADC_DATA) InputRateLimit = 3000.;
		else if(ap->InputType == RTUSB3000::TTL_DATA) InputRateLimit = 6000.;
	}
	else if(UsbSpeed >= RTUSB3000::INVALID_USB_SPEED) { LastErrorNumber = 57; return FALSE; }

	//    
	DspClockout = RTUSB3000::DSP_CLOCK_OUT; //   DSP  
	if(fabs(DspClockout - 72000.0) > 0.00001) { LastErrorNumber = 30; return FALSE; }
	ap->InputRate = fabs(ap->InputRate);
	if(ap->InputRate > InputRateLimit) ap->InputRate = InputRateLimit;
	if(ap->InputRate < 0.1) ap->InputRate = 0.1;
	double SCLOCK_DIV = DspClockout/(2.*ap->InputRate) - 0.5;
	if(SCLOCK_DIV > 65500.) SCLOCK_DIV = 65500.;
	ap->InputRate = DspClockout/(2.*(WORD)(SCLOCK_DIV+1.0));
	if(!PUT_VAR_WORD(RTUSB3000::D_FIRST_SAMPLE_DELAY, (SHORT)(DspClockout/ap->InputRate + 5.5))) { LastErrorNumber = 21; return FALSE; }
	if(!PUT_VAR_WORD(RTUSB3000::D_INPUT_RATE, (SHORT)SCLOCK_DIV)) { LastErrorNumber = 21; return FALSE; }
	ap->InterKadrDelay = 0.0;			//     :(((
	ap->InterKadrDelay = fabs(ap->InterKadrDelay);
	if((1./(ap->InputRate)) > (ap->InterKadrDelay)) ap->InterKadrDelay = 1./ap->InputRate;
	double kadrdelay = ap->InterKadrDelay*ap->InputRate - 0.5;
	if(kadrdelay > 65500.) kadrdelay = 65500.;
	ap->InterKadrDelay = (WORD)(kadrdelay + 1.)/ap->InputRate;
	if(!PUT_VAR_WORD(RTUSB3000::D_INTER_KADR_DELAY, (SHORT)kadrdelay)) { LastErrorNumber = 21; return FALSE; }
	ap->ChannelRate = 1.0/((ap->ChannelsQuantity - 1.0)/ap->InputRate + ap->InterKadrDelay);

	//=====  FIFO   =====
	//   FIFO     0x0
	ap->InputFifoBaseAddress = 0x0;
	//  FIFO        0x400(1024)
	if(ap->InputFifoLength < 0x400) ap->InputFifoLength = 0x400;
	//    0x3000 (12288)
	else if(ap->InputFifoLength > 0x3000) ap->InputFifoLength = 0x3000;
	//    FIFO     ...
	if(ap->InputFifoLength > 0x800) ap->InputFifoLength -= (WORD)(ap->InputFifoLength%2048);
	else ap->InputFifoLength -= (WORD)(ap->InputFifoLength%1024);
	//    DSP   FIFO   
	if(!PUT_VAR_WORD(RTUSB3000::D_INPUT_FIFO_LENGTH, ap->InputFifoLength)) { LastErrorNumber = 21; return FALSE; }

	//     
	for(i = 0x0; i < 0x8; i++)
	{
		if(ap->AdcOffsetCoef[i] >= 0) AdcOffsetCoef = (SHORT)(ap->AdcOffsetCoef[i] + 0.5);
		else AdcOffsetCoef = (SHORT)(ap->AdcOffsetCoef[i] - 0.5);
		if(!PUT_VAR_WORD((WORD)(RTUSB3000::D_ADC_ZERO + i), AdcOffsetCoef)) { LastErrorNumber = 21; return FALSE; }

		AdcScaleCoef = (WORD)(ap->AdcScaleCoef[i] * (double)0x8000 + 0.5);
		if(!PUT_VAR_WORD((WORD)(RTUSB3000::D_ADC_SCALE + i), AdcScaleCoef)) { LastErrorNumber = 21; return FALSE; }
	}

	//         RTUSB3000::INPUT_PARS
	this->ap = *ap;

	//     -       ,
	// ..         
	//   . START_READ(), READ_KADR()  ReadData()
	this->ap.InputEnabled = TRUE;
	// !!!
	return TRUE;
}

//------------------------------------------------------------------------
//    
//------------------------------------------------------------------------
BOOL WINAPI CRTUSB3000::START_READ(void)
{
	WORD InBuf[4];
	DWORD cbRet = 0;

	//   ?
	if(hDevice == INVALID_HANDLE_VALUE) { LastErrorNumber = 4; return FALSE; }
	//          
	//        
//	else if(this->ap.InputEnabled == FALSE)  { LastErrorNumber = 47; return FALSE; }

	//   
	EnterCriticalSection(&cs);

	//        ( AVR)
	InBuf[0] = 0; InBuf[1] = V_START_READ; InBuf[2] = (WORD)(0x0 | DSP_DM); InBuf[3] = (WORD)(ap.InputFifoLength/2.0);
	//      ( AVR)
	if(!DeviceIoControl(hDevice, DIOC_SEND_COMMAND, &InBuf, sizeof(InBuf), NULL, 0, &cbRet, NULL)) { LastErrorNumber = 5; LeaveCriticalSection(&cs); return FALSE; }

	//   
	LeaveCriticalSection(&cs);

	//        DSP 
	if(!SEND_COMMAND(C_START_READ)) { LastErrorNumber = 23; return FALSE; }
	else return TRUE;
}

//------------------------------------------------------------------------
//    
//------------------------------------------------------------------------
BOOL WINAPI CRTUSB3000::STOP_READ(void)
{
	DWORD cbRet=0;

	//   ?
	if(hDevice == INVALID_HANDLE_VALUE) { LastErrorNumber = 4; return FALSE; }
	//    DSP    
	else if(!SEND_COMMAND(C_STOP_READ)) { LastErrorNumber = 23; return FALSE; }

	//   
	EnterCriticalSection(&cs);

	//     bulk'
	if(!DeviceIoControl(hDevice, DIOC_RESET_PIPE3/*reset Read Pipe*/, NULL, NULL, NULL, NULL, &cbRet, NULL)) { LastErrorNumber = 5; LeaveCriticalSection(&cs); return FALSE; }

	//   
	LeaveCriticalSection(&cs);

	return TRUE;
}

//------------------------------------------------------------------------
//       
//------------------------------------------------------------------------
BOOL WINAPI CRTUSB3000::READ_KADR(SHORT * const Data)
{
	WORD ChannelsQuantity;

	//   ?
	if(hDevice == INVALID_HANDLE_VALUE) { LastErrorNumber = 4; return FALSE; }
	else if(strcmp((char *)fi.Name, "USB3000")) { LastErrorNumber = 28; return FALSE; }
	//  
	else if(!STOP_READ()) { LastErrorNumber = 32; return FALSE; }
	//          
	//        
	else if(this->ap.InputEnabled == FALSE)  { LastErrorNumber = 47; return FALSE; }
	//     (  )
	else if(!GET_VAR_WORD(RTUSB3000::D_CONTROL_TABLE_LENGHT, (SHORT *)&ChannelsQuantity)) { LastErrorNumber = 19; return FALSE; }
	else if(ChannelsQuantity != ap.ChannelsQuantity)  { LastErrorNumber = 33; return FALSE; }
	else if(ChannelsQuantity > ap.InputFifoLength)  { LastErrorNumber = 34; return FALSE; }
	//    DSP      
	else if(!SEND_COMMAND(C_READ_KADR)) { LastErrorNumber = 23; return FALSE; }
	//     
	else if(!GET_DM_ARRAY(0x0, ap.ChannelsQuantity, Data)) { LastErrorNumber = 35; return FALSE; }

	return TRUE;
}

//------------------------------------------------------------------------
//       
//------------------------------------------------------------------------
BOOL WINAPI CRTUSB3000::READ_SAMPLE(WORD Channel, SHORT * const Sample)
{
	//   ?
	if(hDevice == INVALID_HANDLE_VALUE) { LastErrorNumber = 4; return FALSE; }
	//  
	else if(!STOP_READ()) { LastErrorNumber = 32; return FALSE; }
	//       
	else if(!PUT_VAR_WORD(RTUSB3000::D_INPUT_CHANNEL, Channel)) { LastErrorNumber = 21; return FALSE; }
	//    DSP     
	else if(!SEND_COMMAND(C_READ_SAMPLE)) { LastErrorNumber = 23; return FALSE; }
	//      
	else if(!GET_VAR_WORD(RTUSB3000::D_INPUT_SAMPLE, Sample)) { LastErrorNumber = 19; return FALSE; }

	return TRUE;
}

//------------------------------------------------------------------------
//     
//------------------------------------------------------------------------
BOOL WINAPI CRTUSB3000::ReadData(SHORT * const lpBuffer, DWORD * const nNumberOfWordsToRead, LPDWORD lpNumberOfBytesRead, LPOVERLAPPED lpOverlapped)
{
	BOOL status;

	//   ?
	if(hDevice == INVALID_HANDLE_VALUE) { LastErrorNumber = 4; return FALSE; }
	//        0x200(512)
	else if(*nNumberOfWordsToRead < 0x200) *nNumberOfWordsToRead = 0x200;
	//    0x100000(1024*1024)
	else if(*nNumberOfWordsToRead > (1024*1024)) *nNumberOfWordsToRead = 1024*1024;
	//       0x200(512)
	*nNumberOfWordsToRead -= (*nNumberOfWordsToRead)%512;

	//         
	status = ReadFile(hDevice, (LPVOID)lpBuffer, 2*(*nNumberOfWordsToRead), lpNumberOfBytesRead, lpOverlapped);

	//   :)))))
	if(status) return TRUE;
	//   ReadFile()
	else { LastErrorNumber = 53; return FALSE; }
}




//========================================================================
//           
//========================================================================

//------------------------------------------------------------------------
//         
//------------------------------------------------------------------------
BOOL WINAPI CRTUSB3000::GET_OUTPUT_PARS(RTUSB3000::OUTPUT_PARS * const dp)
{
	WORD SclkDiv;
	double DspClockout;

	//   ?
	if(hDevice == INVALID_HANDLE_VALUE) { LastErrorNumber = 4; return FALSE; }
	else if(this->dp.size != dp->size) { LastErrorNumber = 36; return FALSE; }
	else if(strcmp((char *)fi.Name, "USB3000")) { LastErrorNumber = 28; return FALSE; }
	else if(!GET_VAR_WORD(RTUSB3000::D_OUTPUT_ENABLED, (SHORT *)&dp->OutputEnabled)) { LastErrorNumber = 19; return FALSE; }

	//    
	DspClockout = RTUSB3000::DSP_CLOCK_OUT; //   DSP  
	if(fabs(DspClockout - 72000.0) > 0.00001) { LastErrorNumber = 30; return FALSE; }

	if(!GET_VAR_WORD(RTUSB3000::D_OUTPUT_SCLK_DIV, (SHORT *)&SclkDiv)) { LastErrorNumber = 19; return FALSE; }
	double SCLK = DspClockout/(2.0*(SclkDiv + 1.0));

	WORD RFS_DIV;
	if(!GET_VAR_WORD(RTUSB3000::D_OUTPUT_RATE, (SHORT *)&RFS_DIV)) { LastErrorNumber = 19; return FALSE; }
	dp->OutputRate = SCLK/(RFS_DIV + 1.0);

	//  FIFO  
	if(!GET_VAR_WORD(RTUSB3000::D_OUTPUT_FIFO_BASE_ADDRESS, (SHORT *)&dp->OutputFifoBaseAddress)) { LastErrorNumber = 19; return FALSE; }
	if(!GET_VAR_WORD(RTUSB3000::D_CUR_OUTPUT_FIFO_LENGTH, (SHORT *)&dp->OutputFifoLength)) { LastErrorNumber = 19; return FALSE; }

	//         RTUSB3000::OUTPUT_PARS
	this->dp = *dp;

	return TRUE;
}

//------------------------------------------------------------------------
//         
//------------------------------------------------------------------------
BOOL WINAPI CRTUSB3000::SET_OUTPUT_PARS(RTUSB3000::OUTPUT_PARS * const dp)
{
	WORD SclkDiv;
	double DspClockout, SCLK;

	//   ?
	if(hDevice == INVALID_HANDLE_VALUE) { LastErrorNumber = 4; return FALSE; }
	else if(this->dp.size != dp->size) { LastErrorNumber = 36; return FALSE; }
	else if(strcmp((char *)fi.Name, "USB3000")) { LastErrorNumber = 28; return FALSE; }

	//    
	DspClockout = RTUSB3000::DSP_CLOCK_OUT; //   DSP  
	if(fabs(DspClockout - 72000.0) > 0.00001) { LastErrorNumber = 30; return FALSE; }

	if(!GET_VAR_WORD(RTUSB3000::D_OUTPUT_SCLK_DIV, (SHORT *)&SclkDiv)) { LastErrorNumber = 19; return FALSE; }
	SCLK = DspClockout/(2.0*(SclkDiv + 1.0));

	dp->OutputRate = fabs(dp->OutputRate);
	if(dp->OutputRate < SCLK/65535.0) dp->OutputRate = SCLK/65535.0;

	WORD RFS_DIV = (SHORT)(SCLK/(dp->OutputRate)-0.5);
	dp->OutputRate = SCLK/(RFS_DIV + 1.0);
	if(!PUT_VAR_WORD(RTUSB3000::D_OUTPUT_RATE, RFS_DIV)) { LastErrorNumber = 21; return FALSE; }

	//=====  FIFO   =====
	//   FIFO     0x3000
	dp->OutputFifoBaseAddress = 0x3000;
	//  FIFO        0x80(128)
	if(dp->OutputFifoLength < 0x80) dp->OutputFifoLength = 0x80;
	//    0xF80(3968)
	else if(dp->OutputFifoLength > 0xF80) dp->OutputFifoLength = 0xF80;
	//    FIFO      0x80(128)
	dp->OutputFifoLength -= (WORD)(dp->OutputFifoLength%128);
	if(!PUT_VAR_WORD(RTUSB3000::D_OUTPUT_FIFO_LENGTH, dp->OutputFifoLength)) { LastErrorNumber = 21; return FALSE; }

	//         RTUSB3000::OUTPUT_PARS
	this->dp = *dp;

	return TRUE;
}

//------------------------------------------------------------------------
//    
//------------------------------------------------------------------------
BOOL WINAPI CRTUSB3000::START_WRITE(void)
{
	WORD InBuf[4];
	DWORD cbRet=0;

	//   ?
	if(hDevice == INVALID_HANDLE_VALUE) { LastErrorNumber = 4; return FALSE; }
	//        
	else if(fi.IsDacPresented == FALSE)  { LastErrorNumber = 37; return FALSE; }

	//    
	EnterCriticalSection(&cs);

	//        ( AVR)
	InBuf[0] = 0; InBuf[1] = V_START_WRITE; InBuf[2] = (WORD)(0x3000 | DSP_DM); InBuf[3] = (WORD)(dp.OutputFifoLength/2.0);
	//      ( AVR)
	if(!DeviceIoControl(hDevice, DIOC_SEND_COMMAND, &InBuf, sizeof(InBuf), NULL, 0, &cbRet, NULL)) { LastErrorNumber = 5; LeaveCriticalSection(&cs); return FALSE; }

	//    
	LeaveCriticalSection(&cs);

	//      DSP    
	if(!SEND_COMMAND(C_START_WRITE)) { LastErrorNumber = 23; return FALSE; }
	else return TRUE;
}

//------------------------------------------------------------------------
//   
//------------------------------------------------------------------------
BOOL WINAPI CRTUSB3000::STOP_WRITE(void)
{
	DWORD cbRet=0;

	//   ?
	if(hDevice == INVALID_HANDLE_VALUE) { LastErrorNumber = 4; return FALSE; }
	//        
	else if(fi.IsDacPresented == FALSE)  { LastErrorNumber = 37; return FALSE; }
	//    DSP    
	else if(!SEND_COMMAND(C_STOP_WRITE)) { LastErrorNumber = 23; return FALSE; }

	//    
	EnterCriticalSection(&cs);

	//     bulk'
	if(!DeviceIoControl(hDevice, DIOC_RESET_PIPE1/*reset Write Pipe*/, NULL, NULL, NULL, NULL, &cbRet, NULL)) { LastErrorNumber = 5; LeaveCriticalSection(&cs); return FALSE; }

	//    
	LeaveCriticalSection(&cs);

	return TRUE;
}

//------------------------------------------------------------------------
//      
//------------------------------------------------------------------------
BOOL WINAPI CRTUSB3000::WriteData(SHORT * const lpBuffer, DWORD * const nNumberOfWordsToWrite, LPDWORD lpNumberOfBytesWritten, LPOVERLAPPED lpOverlapped)
{
	BOOL status;

	//   ?
	if(hDevice == INVALID_HANDLE_VALUE) { LastErrorNumber = 4; return FALSE; }
	//        0x40(64)
	else if(*nNumberOfWordsToWrite < 0x40) *nNumberOfWordsToWrite = 0x40;
	//    0x100000(1024*1024)
	else if(*nNumberOfWordsToWrite > (1024*1024)) *nNumberOfWordsToWrite = 1024*1024;
	//       0x40(64)
	*nNumberOfWordsToWrite -= (*nNumberOfWordsToWrite)%64;

	//         
	status = WriteFile(hDevice, (LPVOID)lpBuffer, 2*(*nNumberOfWordsToWrite), lpNumberOfBytesWritten, lpOverlapped);

	//   :)))))
	if(status) return TRUE;
	//   WriteFile()
	else { LastErrorNumber = 53; return FALSE; }
}

//------------------------------------------------------------------------
//   
//------------------------------------------------------------------------
BOOL WINAPI CRTUSB3000::WRITE_SAMPLE(WORD Channel, SHORT * const Sample)
{
	WORD Data;

	//   ?
	if(hDevice == INVALID_HANDLE_VALUE) { LastErrorNumber = 4; return FALSE; }
	//        
	else if(fi.IsDacPresented == FALSE)  { LastErrorNumber = 37; return FALSE; }
	else if(Channel > 1) { LastErrorNumber = 38; return FALSE; }

	//   
	if((*Sample) < -2048) *Sample = -2048;
	else if((*Sample) > 2047) *Sample = 2047;

	//
	Data = (SHORT)((*Sample) + 2047);
	Data = (WORD)(Data & 0xFFF);
	Data |= (WORD)(Channel << 15) | (WORD)(0x1 << 14);
	if(!PUT_VAR_WORD(RTUSB3000::D_OUTPUT_SAMPLE, Data)) { LastErrorNumber = 21; return FALSE; }

	//    DSP     
	if(!SEND_COMMAND(C_WRITE_SAMPLE)) { LastErrorNumber = 23; return FALSE; }
	return TRUE;
}





//========================================================================
//          
//========================================================================

//************************************************************************
//       
//************************************************************************
BOOL WINAPI CRTUSB3000::ENABLE_TTL_OUT(BOOL EnableTtlOut)
{
	//   ?
	if(hDevice == INVALID_HANDLE_VALUE) { LastErrorNumber = 4; return FALSE; }
	//     
	this->EnableTtlOut = EnableTtlOut;
	//        DSP
	if(!PUT_VAR_WORD(RTUSB3000::D_ENABLE_TTL_OUT, (SHORT)EnableTtlOut)) { LastErrorNumber = 19; return FALSE; }
	//    DSP
	else if(!SEND_COMMAND(C_ENABLE_TTL_OUT)) { LastErrorNumber = 23; return FALSE; }

	return TRUE;
}

//************************************************************************
//        
//************************************************************************
BOOL WINAPI CRTUSB3000::TTL_OUT(WORD TtlOut)
{
	//   ?
	if(hDevice == INVALID_HANDLE_VALUE) { LastErrorNumber = 4; return FALSE; }
	//   /   
	else if(!EnableTtlOut) { LastErrorNumber = 39; return FALSE; }
	//  ,       
	else if(!PUT_VAR_WORD(RTUSB3000::D_TTL_OUT, (SHORT)(TtlOut))) { LastErrorNumber = 21; return FALSE; }
	//    DSP
	else if(!SEND_COMMAND(C_TTL_OUT)) { LastErrorNumber = 23; return FALSE; }

	return TRUE;
}

//************************************************************************
//       
//************************************************************************
BOOL WINAPI CRTUSB3000::TTL_IN(WORD * const TtlIn)
{
	//   ?
	if(hDevice == INVALID_HANDLE_VALUE) { LastErrorNumber = 4; return FALSE; }
	//    DSP
	else if(!SEND_COMMAND(C_TTL_IN)) { LastErrorNumber = 23; return FALSE; }
	//      
	else if(!GET_VAR_WORD(RTUSB3000::D_TTL_IN, (SHORT *)TtlIn)) { LastErrorNumber = 19; return FALSE; }
	//   10      
	*TtlIn = (WORD)((*TtlIn) & 0x03FF);

	return TRUE;
}




//========================================================================
//       
//========================================================================

//------------------------------------------------------------------------
// /     
//------------------------------------------------------------------------
BOOL WINAPI CRTUSB3000::ENABLE_FLASH_WRITE(BOOL EnableFlashWrite)
{
	this->EnableFlashWrite = EnableFlashWrite;

	return TRUE;
}

//------------------------------------------------------------------------
//    FLASH   
//------------------------------------------------------------------------
BOOL WINAPI CRTUSB3000::PUT_FLASH(RTUSB3000::FLASH * const fi)
{
	WORD InBuf[4];
	DWORD cbRet=0;

	//   ?
	if(hDevice == INVALID_HANDLE_VALUE) { LastErrorNumber = 4; return FALSE; }
	//      256 
	else if(fi->size != sizeof(RTUSB3000::FLASH)) { LastErrorNumber = 40; return FALSE; }
	//      
	else if(!EnableFlashWrite) { LastErrorNumber = 41; return FALSE; }

	fi->CRC16 = 0x0;
	fi->CRC16 = CalculateCrc16((BYTE *)(fi) + 2, sizeof(RTUSB3000::FLASH) - 2);

	//    
	EnterCriticalSection(&cs);

	//     
	InBuf[0] = 0; InBuf[1] = V_PUT_FLASH; InBuf[2] = (WORD)(0x1E00); InBuf[3] = 0x0;
	//     
  	if(!DeviceIoControl(hDevice,DIOC_SEND_COMMAND, &InBuf, sizeof(InBuf), fi, 256/*bytes*/, &cbRet, NULL)) { LastErrorNumber = 5; LeaveCriticalSection(&cs); return FALSE; }

	//    
	LeaveCriticalSection(&cs);

	//      RTUSB3000::FLASH
	this->fi = *fi;
	//     
	EnableFlashWrite = FALSE;

	return TRUE;
}

//------------------------------------------------------------------------
//    FLASH   
//------------------------------------------------------------------------
BOOL WINAPI CRTUSB3000::GET_FLASH(RTUSB3000::FLASH * const fi)
{
	BYTE *ptr;
	WORD InBuf[4];
	WORD crc;
	DWORD cbRet=0;

	//   ?
	if(hDevice == INVALID_HANDLE_VALUE) { LastErrorNumber = 4; return FALSE; }
	//      256 
	else if(fi->size != sizeof(RTUSB3000::FLASH))  { LastErrorNumber = 40; return FALSE; }

	//       256 
	ptr = new BYTE[256];
	if(!ptr) { LastErrorNumber = 54; return FALSE; }

	//    
	EnterCriticalSection(&cs);

	//     
	InBuf[0] = 1; InBuf[1] = V_GET_FLASH; InBuf[2] = (WORD)(0x1E00); InBuf[3] = 0x0;
	//     
  	if(!DeviceIoControl(hDevice,DIOC_SEND_COMMAND, &InBuf, sizeof(InBuf), ptr, 256/*bytes*/, &cbRet, NULL)) { LastErrorNumber = 5; delete[] ptr; LeaveCriticalSection(&cs); return FALSE; }

	//    
	LeaveCriticalSection(&cs);

	//  CRC     
	crc = (WORD)(((WORD)ptr[1] << 8) | ptr[0]);
	if(crc != CalculateCrc16((BYTE *)(ptr)+2, sizeof(RTUSB3000::FLASH)-2)) { LastErrorNumber = 55; delete[] ptr; return FALSE; }
	//    -     RTUSB3000::FLASH
	memcpy(fi, ptr, sizeof(RTUSB3000::FLASH));
	//      RTUSB3000::FLASH
	this->fi = *fi;

	//  
	delete[] ptr;

	return TRUE;
}






//========================================================================
//     DSP 
//========================================================================

//------------------------------------------------------------------------
//          DSP
//------------------------------------------------------------------------
BOOL WINAPI CRTUSB3000::PUT_VAR_WORD(WORD Address, SHORT Data)
{
	//   ?
	if(hDevice == INVALID_HANDLE_VALUE) { LastErrorNumber = 4; return FALSE; }
	else if((Address < VarsBaseAddress) || (Address > ProgramBaseAddress)) { LastErrorNumber = 42; return FALSE; }
	else if(!PUT_PM_WORD(Address, (LONG)Data<<8)) { LastErrorNumber = 43; return FALSE; }
	return TRUE;
}

//------------------------------------------------------------------------
//           DSP
//------------------------------------------------------------------------
BOOL WINAPI CRTUSB3000::GET_VAR_WORD(WORD Address, SHORT * const Data)
{
	LONG PmData;

	//   ?
	if(hDevice == INVALID_HANDLE_VALUE) { LastErrorNumber = 4; return FALSE; }
	else if((Address < VarsBaseAddress) || (Address > ProgramBaseAddress)) { LastErrorNumber = 42; return FALSE; }
	else if(!GET_PM_WORD(Address, &PmData)) { LastErrorNumber = 44; return FALSE; }
	*Data = (SHORT)(PmData >> 8);
	return TRUE;
}

//------------------------------------------------------------------------
//        DSP
//------------------------------------------------------------------------
BOOL WINAPI CRTUSB3000::PUT_DM_WORD(WORD Address, SHORT Data)
{
	BOOL status;
	WORD InBuf[4];
	DWORD cbRet=0;

	//   ?
	if(hDevice == INVALID_HANDLE_VALUE) { LastErrorNumber = 4; return FALSE; }

	//    
	EnterCriticalSection(&cs);

   //        DSP
	InBuf[0] = 0; InBuf[1] = V_PUT_ARRAY; InBuf[2] = (WORD)(Address | DSP_DM); InBuf[3] = 0x0;
	//   
	status = DeviceIoControl( hDevice,DIOC_SEND_COMMAND,
                             &InBuf,sizeof(InBuf),
                             &Data, 2/*bytes*/,
                             &cbRet,NULL);

	//    
	LeaveCriticalSection(&cs);

	//   :)))))
	if(status) return TRUE;
	//   DeviceIoControl()
	else { LastErrorNumber = 5; return FALSE; }
}

//------------------------------------------------------------------------
//        DSP
//------------------------------------------------------------------------
BOOL WINAPI CRTUSB3000::GET_DM_WORD(WORD Address, SHORT * const Data)
{
	BOOL status;
	WORD InBuf[4];
	DWORD cbRet=0;

	//   ?
	if(hDevice == INVALID_HANDLE_VALUE) { LastErrorNumber = 4; return FALSE; }

	//    
	EnterCriticalSection(&cs);

   //        DSP
	InBuf[0] = 1; InBuf[1] = V_GET_ARRAY; InBuf[2] = (WORD)(Address | DSP_DM); InBuf[3] = 0x0;
	//   
	status = DeviceIoControl( hDevice,DIOC_SEND_COMMAND,
                             &InBuf,sizeof(InBuf),
                             Data, 2/*bytes*/,
                             &cbRet,NULL);

	//    
	LeaveCriticalSection(&cs);

	//   :)))))
	if(status) return TRUE;
	//   DeviceIoControl()
	else { LastErrorNumber = 5; return FALSE; }
}

//------------------------------------------------------------------------
//        DSP
//------------------------------------------------------------------------
BOOL WINAPI CRTUSB3000::PUT_PM_WORD(WORD Address, LONG Data)
{
	BOOL status;
	WORD InBuf[4];
	DWORD cbRet=0;

	//   ?
	if(hDevice == INVALID_HANDLE_VALUE) { LastErrorNumber = 4; return FALSE; }

	//    
	EnterCriticalSection(&cs);

   //        DSP
	InBuf[0] = 0; InBuf[1] = V_PUT_ARRAY; InBuf[2] = (WORD)(Address | DSP_PM); InBuf[3] = 0x0;
	//   
	status = DeviceIoControl( hDevice,DIOC_SEND_COMMAND,
                             &InBuf, sizeof(InBuf),
                             &Data, 4/*bytes*/,
                             &cbRet,NULL);

	//    
	LeaveCriticalSection(&cs);

	//   :)))))
	if(status) return TRUE;
	//   DeviceIoControl()
	else { LastErrorNumber = 5; return FALSE; }
}

//------------------------------------------------------------------------
//        DSP
//------------------------------------------------------------------------
BOOL WINAPI CRTUSB3000::GET_PM_WORD(WORD Address, LONG * const Data)
{
	BOOL status;
	WORD InBuf[4];
	DWORD cbRet=0;

	//   ?
	if(hDevice == INVALID_HANDLE_VALUE) { LastErrorNumber = 4; return FALSE; }

	//    
	EnterCriticalSection(&cs);

   //        DSP
	InBuf[0] = 1; InBuf[1] = V_GET_ARRAY; InBuf[2] = (WORD)(Address | DSP_PM); InBuf[3] = 0x0;
	//   
	status = DeviceIoControl( hDevice,DIOC_SEND_COMMAND,
                             &InBuf,sizeof(InBuf),
                             Data, 4/*bytes*/,
                             &cbRet,NULL);

	//    
	LeaveCriticalSection(&cs);

	//   :)))))
	if(status) return TRUE;
	//   DeviceIoControl()
	else { LastErrorNumber = 5; return FALSE; }
}

//------------------------------------------------------------------------
//         DSP
//------------------------------------------------------------------------
BOOL WINAPI CRTUSB3000::PUT_DM_ARRAY(WORD BaseAddress, WORD NPoints, SHORT * const Data)
{
	BOOL error = FALSE;
	WORD InBuf[4];
	WORD BlockQuantity, RestDataPoints;
	DWORD cbRet = 0x0;

	//   ?
	if(hDevice == INVALID_HANDLE_VALUE) { LastErrorNumber = 4; return FALSE; }

	BlockQuantity = (WORD)((sizeof(SHORT)*NPoints)/MAX_USB_BLOCK);
	RestDataPoints = (WORD)((sizeof(SHORT)*NPoints)%MAX_USB_BLOCK); //  

	InBuf[0] = 0; InBuf[1] = V_PUT_ARRAY; InBuf[3] = 0x0;
	for(WORD i=0; i < BlockQuantity; i++)
	{
		InBuf[2] = (WORD)((BaseAddress + i*MAX_USB_BLOCK/sizeof(SHORT)) | DSP_DM);
		EnterCriticalSection(&cs);
		if(!DeviceIoControl(hDevice,DIOC_SEND_COMMAND, &InBuf, sizeof(InBuf), Data + i*MAX_USB_BLOCK/sizeof(SHORT), MAX_USB_BLOCK/*bytes*/, &cbRet, NULL)) error = TRUE;
		LeaveCriticalSection(&cs);
      if(error) break;
	}

	if(!error && RestDataPoints)
	{
		InBuf[2] = (WORD)((BaseAddress + BlockQuantity*MAX_USB_BLOCK/sizeof(SHORT)) | DSP_DM);
		EnterCriticalSection(&cs);
		if(!DeviceIoControl(hDevice,DIOC_SEND_COMMAND, &InBuf, sizeof(InBuf), Data + BlockQuantity*MAX_USB_BLOCK/sizeof(SHORT), RestDataPoints/*bytes*/, &cbRet, NULL)) error = TRUE;
		LeaveCriticalSection(&cs);
	}

	if(error) { LastErrorNumber = 5; return FALSE; }
	return TRUE;
}

//------------------------------------------------------------------------
//         DSP
//------------------------------------------------------------------------
BOOL WINAPI CRTUSB3000::GET_DM_ARRAY(WORD BaseAddress, WORD NPoints, SHORT * const Data)
{
	BOOL error = FALSE;
	WORD InBuf[4];
	WORD BlockQuantity, RestDataPoints;
	DWORD cbRet = 0x0;

	//   ?
	if(hDevice == INVALID_HANDLE_VALUE) { LastErrorNumber = 4; return FALSE; }

	BlockQuantity = (WORD)((sizeof(SHORT)*NPoints)/MAX_USB_BLOCK);
	RestDataPoints = (WORD)((sizeof(SHORT)*NPoints)%MAX_USB_BLOCK); //  

	InBuf[0] = 1; InBuf[1] = V_GET_ARRAY; InBuf[3] = 0x0;
	for(WORD i=0; i < BlockQuantity; i++)
	{
		InBuf[2] = (WORD)((BaseAddress + i*MAX_USB_BLOCK/sizeof(SHORT)) | DSP_DM);
		EnterCriticalSection(&cs);
		if(!DeviceIoControl( hDevice, DIOC_SEND_COMMAND, &InBuf,sizeof(InBuf), Data + i*MAX_USB_BLOCK/sizeof(SHORT), MAX_USB_BLOCK/*bytes*/, &cbRet, NULL)) error = TRUE;
		LeaveCriticalSection(&cs);
      if(error) break;
	}

	if(!error && RestDataPoints)
	{
		InBuf[2] = (WORD)((BaseAddress + BlockQuantity*MAX_USB_BLOCK/sizeof(SHORT)) | DSP_DM);
		EnterCriticalSection(&cs);
		if(!DeviceIoControl(hDevice, DIOC_SEND_COMMAND, &InBuf, sizeof(InBuf), Data + BlockQuantity*MAX_USB_BLOCK/sizeof(SHORT), RestDataPoints/*bytes*/, &cbRet, NULL)) error = TRUE;
		LeaveCriticalSection(&cs);
	}

	if(error) { LastErrorNumber = 5; return FALSE; }
	return TRUE;
}

//------------------------------------------------------------------------
//         DSP
//------------------------------------------------------------------------
BOOL WINAPI CRTUSB3000::PUT_PM_ARRAY(WORD BaseAddress, WORD NPoints, LONG * const Data)
{
	BOOL error = FALSE;
	WORD InBuf[4];
	WORD BlockQuantity, RestDataPoints;
	DWORD cbRet=0;

	//   ?
	if(hDevice == INVALID_HANDLE_VALUE) { LastErrorNumber = 4; return FALSE; }

	BlockQuantity = (WORD)((sizeof(LONG)*NPoints)/MAX_USB_BLOCK);
	RestDataPoints = (WORD)((sizeof(LONG)*NPoints)%MAX_USB_BLOCK); //  

	InBuf[0] = 0; InBuf[1] = V_PUT_ARRAY; InBuf[3] = 0x0;
	for(WORD i=0; i < BlockQuantity; i++)
	{
		InBuf[2] = (WORD)((BaseAddress + i*MAX_USB_BLOCK/sizeof(LONG)) | DSP_PM);
		EnterCriticalSection(&cs);
		if(!DeviceIoControl(hDevice, DIOC_SEND_COMMAND, &InBuf, sizeof(InBuf), Data + i*MAX_USB_BLOCK/sizeof(LONG), MAX_USB_BLOCK/*bytes*/, &cbRet, NULL)) error = TRUE;
		LeaveCriticalSection(&cs);
      if(error) break;
	}

	if(!error && RestDataPoints)
	{
		InBuf[2] = (WORD)((BaseAddress + BlockQuantity*MAX_USB_BLOCK/sizeof(LONG)) | DSP_PM);
		EnterCriticalSection(&cs);
		if(!DeviceIoControl(hDevice, DIOC_SEND_COMMAND, &InBuf, sizeof(InBuf), Data + BlockQuantity*MAX_USB_BLOCK/sizeof(LONG), RestDataPoints/*bytes*/, &cbRet, NULL)) error = TRUE;
		LeaveCriticalSection(&cs);
	}

	if(error) { LastErrorNumber = 5; return FALSE; }
	return TRUE;
}

//------------------------------------------------------------------------
//         DSP
//------------------------------------------------------------------------
BOOL WINAPI CRTUSB3000::GET_PM_ARRAY(WORD BaseAddress, WORD NPoints, LONG * const Data)
{
	BOOL error = FALSE;
	WORD InBuf[4];
	WORD BlockQuantity, RestDataPoints;
	DWORD cbRet=0;

	//   ?
	if(hDevice == INVALID_HANDLE_VALUE) { LastErrorNumber = 4; return FALSE; }

	BlockQuantity = (WORD)((sizeof(LONG)*NPoints)/MAX_USB_BLOCK);
	RestDataPoints = (WORD)((sizeof(LONG)*NPoints)%MAX_USB_BLOCK); //  

	InBuf[0] = 1; InBuf[1] = V_GET_ARRAY; InBuf[3] = 0x0;
	for(WORD i=0; i < BlockQuantity; i++)
	{
		InBuf[2] = (WORD)((BaseAddress + i*MAX_USB_BLOCK/sizeof(LONG)) | DSP_PM);
		EnterCriticalSection(&cs);
		if(!DeviceIoControl(hDevice, DIOC_SEND_COMMAND, &InBuf, sizeof(InBuf), Data + i*MAX_USB_BLOCK/sizeof(LONG), MAX_USB_BLOCK/*bytes*/, &cbRet, NULL)) error = TRUE;
		LeaveCriticalSection(&cs);
      if(error) break;
	}

	if(!error && RestDataPoints)
	{
		InBuf[2] = (WORD)((BaseAddress + BlockQuantity*MAX_USB_BLOCK/sizeof(LONG)) | DSP_PM);
		EnterCriticalSection(&cs);
	   if(!DeviceIoControl(hDevice, DIOC_SEND_COMMAND, &InBuf, sizeof(InBuf), Data + BlockQuantity*MAX_USB_BLOCK/sizeof(LONG), RestDataPoints/*bytes*/, &cbRet, NULL)) error = TRUE;
		LeaveCriticalSection(&cs);
	}

	if(error) { LastErrorNumber = 5; return FALSE; }
	return TRUE;
}



//------------------------------------------------------------------------------
//     Windows
//------------------------------------------------------------------------------
DWORD CRTUSB3000::GetWindowsVersion(void)
{
	OSVERSIONINFO osvi;

	ZeroMemory(&osvi, sizeof(OSVERSIONINFO));
	osvi.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);

	if(!GetVersionEx(&osvi))
	{
		osvi.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);
		if(!GetVersionEx((OSVERSIONINFO *)&osvi)) return UNKNOWN_WINDOWS_VERSION;
	}

	switch (osvi.dwPlatformId)
	{
		case VER_PLATFORM_WIN32_NT:
			if(osvi.dwMajorVersion <= 4 ) return WINDOWS_NT; 		// WinNT
			else if((osvi.dwMajorVersion == 5) && (osvi.dwMinorVersion == 0)) return WINDOWS_2000; 			// Win2000
			else if((osvi.dwMajorVersion == 5) && (osvi.dwMinorVersion == 1)) return WINDOWS_XP; 				// WinXP
			else if((osvi.dwMajorVersion == 5) && (osvi.dwMinorVersion == 2)) return WINDOWS_2003_SERVER; 	// Win2003 Server
			else if(osvi.dwMajorVersion == 6) return WINDOWS_VISTA; 	// Vista
			else return UNKNOWN_WINDOWS_VERSION;

		case VER_PLATFORM_WIN32_WINDOWS:
			if((osvi.dwMajorVersion > 0x4) || ((osvi.dwMajorVersion == 0x4)&&(osvi.dwMinorVersion > 0x0))) return WINDOWS_98_OR_LATER; // Win98 or later
			else return WINDOWS_95; 											// Win95

		case VER_PLATFORM_WIN32s: return WINDOWS_32S; 					// Win32s
	}

	return UNKNOWN_WINDOWS_VERSION;
}

//------------------------------------------------------------------------
//  
//------------------------------------------------------------------------
LONG *CRTUSB3000::Transform_Buffer(WORD PmPoints, LONG * const Buffer)
{
	for(WORD i=0; i < PmPoints; i++) Buffer[i] = ((Buffer[i] & 0x0000FFFF) << 8) | ((Buffer[i] & 0x00FF0000) >> 16);
	return Buffer;
}

//------------------------------------------------------------------------
//       
//   :
//              ,
//               
//------------------------------------------------------------------------
BOOL CRTUSB3000::FindoutInputType(RTUSB3000::INPUT_PARS * const ap)
{
	WORD i;

	ap->InputType = RTUSB3000::EMPTY_DATA;
	for(i = 0x0; i < ap->ChannelsQuantity; i++)
	{
		//    
		if(ap->ControlTable[i] & 0x8)	{ ap->InputType |= RTUSB3000::TTL_DATA; break; }
		//   
		else ap->InputType |= RTUSB3000::ADC_DATA;
	}
	return TRUE;
}

//------------------------------------------------------------------------
//     
//       :(((
//------------------------------------------------------------------------
#pragma argsused
BOOL CRTUSB3000::ControlTableCreation(RTUSB3000::INPUT_PARS * const ap)
{
	return TRUE;
}

//----------------------------------------------------------------
//     CRC16
//----------------------------------------------------------------
WORD CRTUSB3000::CalculateCrc16(BYTE *ptr, WORD NBytes)
{
	WORD w = 0x0;
	WORD i, j;

	for(i = 0x0; i < NBytes; i++)
	{
		w ^= (WORD)((WORD)(ptr[i]) << 0x8);
		for(j = 0x0; j < 8; j++)
		{
			if(w & (WORD)0x8000) w = (WORD)((w << 0x1) ^ 0x8005);
			else w <<= 0x1;
		}
	}
	return w;
}

